<?php
/*
 * ----------------------------------------------------------------------
 *
 *                          Borlabs Cookie
 *                      developed by Borlabs
 *
 * ----------------------------------------------------------------------
 *
 * Copyright 2018 Borlabs - Benjamin A. Bornschein. All rights reserved.
 * This file may not be redistributed in whole or significant part.
 * Content of this file is protected by international copyright laws.
 *
 * ----------------- Borlabs Cookie IS NOT FREE SOFTWARE -----------------
 *
 * @copyright Borlabs - Benjamin A. Bornschein, https://borlabs.io
 * @author Benjamin A. Bornschein, Borlabs ben@borlabs.io
 *
 */

namespace BorlabsCookie\Cookie\Backend;

use BorlabsCookie\Cookie\API;
use BorlabsCookie\Cookie\Config;
use BorlabsCookie\Cookie\Multilanguage;
use BorlabsCookie\Cookie\Tools;

class Settings
{
    private static $instance;

    private $imagePath;

    public static function getInstance()
    {
        if (null === self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    private function __clone()
    {
    }

    private function __wakeup()
    {
    }

    protected function __construct()
    {
        global $wpdb;

        // Check if upgrade created the table
        if (get_option('BorlabsCookieTableCreated') === false) {
            if (\BorlabsCookie\Cookie\Install::getInstance()->checkIfTableExists($wpdb->prefix.'borlabs_cookie_log') === false) {
                \BorlabsCookie\Cookie\Upgrade::getInstance()->upgradeVersion_1_5();

                // Check again
                if (\BorlabsCookie\Cookie\Install::getInstance()->checkIfTableExists($wpdb->prefix.'borlabs_cookie_log') === false) {
                    Backend::getInstance()->addMessage(sprintf(_x('The table <strong>%s</strong> could not be created, please check your server error logs for more details.', 'Status message', 'borlabs-cookie'), $wpdb->prefix.'borlabs_cookie_log'), 'error');
                }
            } else {
                update_option('BorlabsCookieTableCreated', true, 'no');
            }
        }

        // Check if tables where installed
        if (get_option('BorlabsCookieBlockedContentTypesTableCreated') === false) {
            if (\BorlabsCookie\Cookie\Install::getInstance()->checkIfTableExists($wpdb->prefix.'borlabs_cookie_blocked_content_types') === false) {

                \BorlabsCookie\Cookie\Upgrade::getInstance()->upgradeVersion_1_9();

                // Check again
                if (\BorlabsCookie\Cookie\Install::getInstance()->checkIfTableExists($wpdb->prefix.'borlabs_cookie_blocked_content_types') === false) {
                    Backend::getInstance()->addMessage(sprintf(_x('The table <strong>%s</strong> could not be created, please check your server error logs for more details.', 'Status message', 'borlabs-cookie'), $wpdb->prefix.'borlabs_cookie_blocked_content_types'), 'error');
                }
            } else {
                update_option('BorlabsCookieBlockedContentTypesTableCreated', true, 'no');
            }
        }

        // Check if table column was changed
        if (get_option('BorlabsCookieTableModified') === false) {
            if (\BorlabsCookie\Cookie\Install::getInstance()->checkTypeOfColumn($wpdb->prefix.'borlabs_cookie_log', 'option', 'varchar') === false) {

                \BorlabsCookie\Cookie\Upgrade::getInstance()->upgradeVersion_1_9();

                // Check again
                if (\BorlabsCookie\Cookie\Install::getInstance()->checkTypeOfColumn($wpdb->prefix.'borlabs_cookie_log', 'option', 'varchar') === false) {
                    Backend::getInstance()->addMessage(sprintf(_x('The table <strong>%s</strong> could not be modified, please check your server error logs for more details.', 'Status message', 'borlabs-cookie'), $wpdb->prefix.'borlabs_cookie_log'), 'error');
                }
            } else {
                update_option('BorlabsCookieTableModified', true, 'no');
            }
        }

        // Check if cache folder exists
        if (!file_exists(WP_CONTENT_DIR.'/cache')) {
            if (!is_writable(WP_CONTENT_DIR)) {
                Backend::getInstance()->addMessage(sprintf(_x('The folder <strong>/%s/</strong> is not writable. Please set the right permissions. See <a href="https://borlabs.io/folder-permissions/" rel="noreferrer" target="_blank">FAQ</a>.', 'Status message', 'borlabs-cookie'), basename(WP_CONTENT_DIR)), 'error');
            } else {
                mkdir(WP_CONTENT_DIR.'/cache');
                mkdir(WP_CONTENT_DIR.'/cache/borlabs_cookie');
            }
        }

        if (file_exists(WP_CONTENT_DIR.'/cache') && !is_writable(WP_CONTENT_DIR.'/cache')) {
            Backend::getInstance()->addMessage(sprintf(_x('The folder <strong>/%s/cache/</strong> is not writable. Please set the right permissions. See <a href="https://borlabs.io/folder-permissions/" rel="noreferrer" target="_blank">FAQ</a>.', 'Status message', 'borlabs-cookie'), basename(WP_CONTENT_DIR)), 'error');
        }

        if (file_exists(WP_CONTENT_DIR.'/cache') && is_writable(WP_CONTENT_DIR.'/cache') && !file_exists(WP_CONTENT_DIR.'/cache/borlabs_cookie')) {
            mkdir(WP_CONTENT_DIR.'/cache/borlabs_cookie');
        }

        if (file_exists(WP_CONTENT_DIR.'/cache/borlabs_cookie') && !is_writable(WP_CONTENT_DIR.'/cache/borlabs_cookie')) {
            Backend::getInstance()->addMessage(sprintf(_x('The folder <strong>/%s/cache/borlabs_cookie/</strong> is not writable. Please set the right permissions. See <a href="https://borlabs.io/folder-permissions/" rel="noreferrer" target="_blank">FAQ</a>.', 'Status message', 'borlabs-cookie'), basename(WP_CONTENT_DIR)), 'error');
        }

        if (!file_exists(WP_CONTENT_DIR.'/cache/borlabs_cookie')) {
            Backend::getInstance()->addMessage(sprintf(_x('The folder <strong>/%s/cache/borlabs_cookie/</strong> does not exist. Please set the right permissions. See <a href="https://borlabs.io/folder-permissions/" rel="noreferrer" target="_blank">FAQ</a>.', 'Status message', 'borlabs-cookie'), basename(WP_CONTENT_DIR)), 'error');
        }

        // Check if HTTPS settings are correct
        $contentURL = parse_url(WP_CONTENT_URL);

        if ($contentURL['scheme'] !== 'https') {
            if ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') || $_SERVER['SERVER_PORT'] === '443') {
                Backend::getInstance()->addMessage(_x('Your SSL configuration is not correct. Please go to <strong>Settings &gt; General</strong> and replace <strong>http://</strong> with <strong>https://</strong> in the settings <strong>WordPress Address (URL)</strong> and <strong>Site Address (URL)</strong>.', 'Status message', 'borlabs-cookie'), 'error');
            }
        }
    }

    /**
     * display function.
     *
     * @access public
     * @return void
     */
    public function display()
    {
        $this->imagePath = plugins_url('images', realpath(__DIR__.'/../../'));

        $tab = !empty($_GET['tab']) ? $_GET['tab'] : 'settings';

        include Backend::getInstance()->templatePath.'/settings.html';

        // Check News
        $lastCheck = intval(get_site_option('BorlabsCookieNewsLastCheck', 0));
        if (empty($lastCheck) || $lastCheck < intval(date('Ymd', mktime(date('H'), date('i'), date('s'), date('m'), date('d')-3)))) {
            $responseNews = API::getInstance()->getNews();
        }

        $borlabsCookieNews = get_site_option('BorlabsCookieNews');

        if (!empty($borlabsCookieNews)) {
            $currentLanguageCode = Multilanguage::getInstance()->getCurrentLanguageCode();

            if (!empty($borlabsCookieNews->{$currentLanguageCode})) {
                Backend::getInstance()->addMessage($borlabsCookieNews->{$currentLanguageCode}, 'offer');
            } else {
                if (!empty($borlabsCookieNews->en)) {
                    Backend::getInstance()->addMessage($borlabsCookieNews->en, 'offer');
                }
            }
        }

        if ($tab == 'statistics') {
            $this->displayTabStatistics();
        } elseif ($tab == 'license') {
            License::getInstance()->displayTabLicense();
        } else {

            // Check if test environment is active
            if (Config::getInstance()->get('testEnvironment') === true) {
                Backend::getInstance()->addMessage(_x('Borlabs Cookie test environment mode is currently activated. <a href="?page=borlabs_cookie_settings&amp;tab=license">Click here</a> to disable the mode.', 'Status message', 'borlabs-cookie'), 'notice');
            }

            // Display language message
            $this->displayLoadedLanguageSettingsMessage();

            if ($tab == 'settings') {
                $this->displayTabSettings();
            }  elseif ($tab == 'blocked_content_types') {
                BlockedContentTypes::getInstance()->displayTabBlockedContentTypes();
            } elseif ($tab == 'texts') {
                $this->displayTabTexts();
            } elseif ($tab == 'styling') {
                $this->displayTabStyling();
            } elseif ($tab == 'import-export') {
                $this->displayTabImportExport();
            } else {
                $this->displayTabSettings();
            }
        }
    }

    /**
     * displayLoadedLanguageSettingsMessage function.
     *
     * @access public
     * @return void
     */
    public function displayLoadedLanguageSettingsMessage()
    {
        // Give info which language setting is loaded
        if (Multilanguage::getInstance()->isMultilanguagePluginActive()) {

            $currentFlag = '';
            $currentLanguage = Multilanguage::getInstance()->getCurrentLanguageName();
            $currentFlagURL = Multilanguage::getInstance()->getCurrentLanguageFlag();

            if (!empty($currentFlagURL)) {
                $currentFlag = '<img src="'.$currentFlagURL.'" alt="'.$currentLanguage.'">';
            }

            Backend::getInstance()->addMessage(sprintf(_x('<span>You are seeing the settings for the language</span> <strong>%s %s</strong>', 'Status message', 'borlabs-cookie'), $currentLanguage, $currentFlag), 'info');
        }
    }

    /**
     * displayTabSettings function.
     *
     * @access public
     * @return void
     */
    public function displayTabSettings()
    {
        if (!empty($_POST['formSend']) && check_admin_referer('borlabs_cookie_settings')) {
            $this->saveSettings($_POST);

            Backend::getInstance()->addMessage(_x('Saved successfully.', 'Status message', 'borlabs-cookie'), 'success');
        }

        // Default network settings
        $siteURLInfo = parse_url(site_url());
        $networkDomain = $siteURLInfo['host'];
        $networkPath = !empty($siteURLInfo['path']) ? $siteURLInfo['path'] : '/';

        $checkboxCookieActivated        = Config::getInstance()->get('cookieActivated') == 'yes' ? ' checked' : '';

        // Option 1
        $textareaCodeCookieThirdParty   = esc_textarea((Config::getInstance()->get('codeCookieThirdParty')));
        $inputGoogleAnalyticsId         = esc_html(!empty(Config::getInstance()->get('googleAnalyticsId')) ? Config::getInstance()->get('googleAnalyticsId') : '');
        $inputFacebookPixelId           = esc_html(!empty(Config::getInstance()->get('facebookPixelId')) ? Config::getInstance()->get('facebookPixelId') : '');
        $inputCookieLifetime            = intval(Config::getInstance()->get('cookieLifetime'));
        $checkboxCookiesForBots         = Config::getInstance()->get('cookiesForBots') ? ' checked' : '';

        // Option 2
        $checkboxShowCookieOption2      = Config::getInstance()->get('showCookieOption2') ? ' checked' : '';
        $textareaCodeFirstParty         = esc_textarea((Config::getInstance()->get('codeCookieFirstParty')));
        $inputCookieLifetimeOption2     = intval(Config::getInstance()->get('cookieLifetimeOption2'));
        $checkboxExecuteFirstPartyCodeOnThirdParty = Config::getInstance()->get('executeFirstPartyCodeOnThirdParty') ? ' checked' : '';

        // Option 3
        $checkboxShowCookieOption3      = Config::getInstance()->get('showCookieOption3') ? ' checked' : '';
        $inputCookieLifetimeOption3     = intval(Config::getInstance()->get('cookieLifetimeOption3'));
        $checkboxCookieRespectDoNotTrack= Config::getInstance()->get('cookieRespectDoNotTrack') ? ' checked' : '';

        // Cookie box settings
        $optionCookieBoxShowAutomatic   = Config::getInstance()->get('cookieBoxShow') === 'automatic' ? ' selected' : '';
        $optionCookieBoxShowAlways      = Config::getInstance()->get('cookieBoxShow') === 'always' ? ' selected' : '';
        $optionCookieBoxShowNever       = Config::getInstance()->get('cookieBoxShow') === 'never' ? ' selected' : '';
        $checkboxCookieBoxBlocks        = Config::getInstance()->get('cookieBoxBlocks') ? ' checked' : '';
        $optionCookieBoxIntegrationScript   = Config::getInstance()->get('cookieBoxIntegration') === 'script' ? ' selected' : '';
        $optionCookieBoxIntegrationHTML     = Config::getInstance()->get('cookieBoxIntegration') === 'html' ? ' selected' : '';
        $optionCookieBoxPositionTop     = Config::getInstance()->get('cookieBoxPosition') === 'top' ? ' selected' : '';
        $optionCookieBoxPositionCenter  = Config::getInstance()->get('cookieBoxPosition') === 'center' ? ' selected' : '';
        $optionCookieBoxPositionBottom  = Config::getInstance()->get('cookieBoxPosition') === 'bottom' ? ' selected' : '';
        $optionPrivacyPageId            = intval(Config::getInstance()->get('privacyPageId'));
        $optionImprintPageId            = intval(Config::getInstance()->get('imprintPageId'));
        $textareaHideCookieBoxOnPage    = implode("\n", Config::getInstance()->get('hideCookieBoxOnPage'));

        // Multisite Network Cookie settings
        $inputCookieDomain              = esc_html(!empty(Config::getInstance()->get('cookieDomain')) ? Config::getInstance()->get('cookieDomain') : $networkDomain);
        $inputCookiePath                = esc_html(!empty(Config::getInstance()->get('cookiePath')) ? Config::getInstance()->get('cookiePath') : $networkPath);
        $checkboxAggregateCookieConsent = Config::getInstance()->get('aggregateCookieConsent') ? ' checked' : '';

        // Check if the installation is not a multisite and the host is different
        $cookieDomainIsDifferent = false;

        if (!defined('WP_ALLOW_MULTISITE') || WP_ALLOW_MULTISITE === false) {
            if (!empty(Config::getInstance()->get('cookieDomain')) && $networkDomain !== Config::getInstance()->get('cookieDomain')) {
                $cookieDomainIsDifferent = true;
            }
        }

        // Iframe block settings
        $checkboxRemoveIframesInFeeds   = Config::getInstance()->get('removeIframesInFeeds') ? ' checked' : '';

        // Check if only one cookie option is available
        if (Config::getInstance()->get('showCookieOption2') === false && Config::getInstance()->get('showCookieOption3') === false) {
            Backend::getInstance()->addMessage(_x('You currently only give the visitor one cookie option to choose from.<br>The ePrivacy does not allow a visitor to be locked out or forced to track. You should therefore provide more than one cookie option.', 'Status message', 'borlabs-cookie'), 'notice');
        }

        include Backend::getInstance()->templatePath.'/settings-cookies.html';
    }

    /**
     * displayTabStatistics function.
     *
     * @access public
     * @return void
     */
    public function displayTabStatistics()
    {
        global $wpdb;

        // Get stats
        $stats = [
            'all'=>0,
            'first-party'=>0,
            'reject-all'=>0,
            'opted-out-ga'=>0,
            'opted-out-fb'=>0,
        ];

        // Get MySQL version, because statistic query only works fast enough with MysqL 5.6 and higher
        $mysqlVersionResult = $wpdb->get_results('
            SELECT
                version() as "version"
        ');

        $hideStats      = true;
        $mysqlVersion   = '-';

        if (!empty($mysqlVersionResult[0]->version)) {
            $mysqlVersion = $mysqlVersionResult[0]->version;

            #if (version_compare($mysqlVersionResult[0]->version, '5.6', '>=')) {
            // Temporary disabled - if MySQL 5.5 users still having issues after the update, the statistics will only be visible to users with MySQL 5.6 and higher
            if (true) {
                $tableName = (Config::getInstance()->get('aggregateCookieConsent') ? $wpdb->base_prefix : $wpdb->prefix).'borlabs_cookie_log';

                // Maintenance
                /*
                $wpdb->get_results('
                    DELETE FROM
                    	`'.$tableName.'`
                    WHERE `uid` IN
                    (
                    	SELECT * FROM
                    	(
                    		SELECT
                    			`uid`
                    		FROM
                    			`'.$tableName.'`
                    		WHERE
                    			`stamp` < NOW() - INTERVAL 7 DAY
                    		GROUP BY
                    			`uid`
                    		HAVING
                    			COUNT(*)=1
                    	) as tmp
                    )
                ');
                */

                $statsResult = $wpdb->get_results('
                    SELECT
                        COUNT(*) as `total`,
                        `option`
                    FROM
                        `'.$tableName.'` as bcl
                    INNER JOIN (
                        SELECT
                            `uid`,
                            MAX(`stamp`) as `mostRecentStamp`
                        FROM
                            `'.$tableName.'`
                        WHERE
                            `stamp` > NOW() - INTERVAL 7 DAY
                        GROUP BY
                            uid
                    ) as mostRecent
                    ON
                        bcl.`uid`=mostRecent.`uid`
                        AND
                        bcl.`stamp`=mostRecent.`mostRecentStamp`
                    WHERE
                        `option` IN ("all","first-party","reject-all")
                    GROUP BY
                        `option`
                ');

                if (!empty($statsResult)) {
                    foreach ($statsResult as $statsData) {
                        $stats[$statsData->option] = $statsData->total;
                    }
                }

                $optedOutStatsResult = $wpdb->get_results('
                    SELECT
                        COUNT(*) as `total`,
                        `option`
                    FROM
                        `'.$tableName.'`
                    WHERE
                        `option` IN ("opted-out-ga", "opted-out-fb")
                        AND
                        `stamp` > NOW() - INTERVAL 7 DAY
                    GROUP BY
                        `option`
                ');

                if (!empty($optedOutStatsResult)) {
                    foreach ($optedOutStatsResult as $statsData) {
                        $stats[$statsData->option] = $statsData->total;
                    }
                }

                $hideStats = false;
            }
        }

        include Backend::getInstance()->templatePath.'/settings-statistics.html';
    }

    /**
     * displayTabTexts function.
     *
     * @access public
     * @return void
     */
    public function displayTabTexts()
    {
        if (!empty($_POST['formSend']) && check_admin_referer('borlabs_cookie_settings')) {
            $this->saveTexts($_POST);

            Backend::getInstance()->addMessage(_x('Saved successfully.', 'Status message', 'borlabs-cookie'), 'success');
        }

        $inputTextCookieOptionThirdParty        = esc_html(!empty(Config::getInstance()->get('textCookieOptionThirdParty')) ? Config::getInstance()->get('textCookieOptionThirdParty') : Config::getInstance()->defaultConfig()['textCookieOptionThirdParty']);
        $inputTextCookieOptionFirstParty        = esc_html(!empty(Config::getInstance()->get('textCookieOptionFirstParty')) ? Config::getInstance()->get('textCookieOptionFirstParty') : Config::getInstance()->defaultConfig()['textCookieOptionFirstParty']);
        $inputTextCookieOptionRejectThirdParty  = esc_html(!empty(Config::getInstance()->get('textCookieOptionRejectThirdParty')) ? Config::getInstance()->get('textCookieOptionRejectThirdParty') : Config::getInstance()->defaultConfig()['textCookieOptionRejectThirdParty']);
        $inputTextHelpHeadline                  = esc_html(!empty(Config::getInstance()->get('textHelpHeadline')) ? Config::getInstance()->get('textHelpHeadline') : Config::getInstance()->defaultConfig()['textHelpHeadline']);
        $textareaTextHelpTop                    = esc_textarea(!empty(Config::getInstance()->get('textHelpTop')) ? Config::getInstance()->get('textHelpTop') : Config::getInstance()->defaultConfig()['textHelpTop']);
        $textareaTextHelpCookieOptionThirdParty = esc_textarea(!empty(Config::getInstance()->get('textHelpCookieOptionThirdParty')) ? Config::getInstance()->get('textHelpCookieOptionThirdParty') : Config::getInstance()->defaultConfig()['textHelpCookieOptionThirdParty']);
        $textareaTextHelpCookieOptionFirstParty = esc_textarea(!empty(Config::getInstance()->get('textHelpCookieOptionFirstParty')) ? Config::getInstance()->get('textHelpCookieOptionFirstParty') : Config::getInstance()->defaultConfig()['textHelpCookieOptionFirstParty']);
        $textareaTextHelpCookieOptionRejectThirdParty = esc_textarea(!empty(Config::getInstance()->get('textHelpCookieOptionRejectThirdParty')) ? Config::getInstance()->get('textHelpCookieOptionRejectThirdParty') : Config::getInstance()->defaultConfig()['textHelpCookieOptionRejectThirdParty']);
        $textareaTextHelpBottom                 = esc_textarea(!empty(Config::getInstance()->get('textHelpBottom')) ? Config::getInstance()->get('textHelpBottom') : Config::getInstance()->defaultConfig()['textHelpBottom']);
        $inputTextTitle                         = esc_html(!empty(Config::getInstance()->get('textTitle')) ? Config::getInstance()->get('textTitle') : Config::getInstance()->defaultConfig()['textTitle']);
        $textareaTextExplanation                = esc_textarea(!empty(Config::getInstance()->get('textExplanation')) ? Config::getInstance()->get('textExplanation') : Config::getInstance()->defaultConfig()['textExplanation']);
        $inputTextInstruction                   = esc_html(!empty(Config::getInstance()->get('textInstruction')) ? Config::getInstance()->get('textInstruction') : Config::getInstance()->defaultConfig()['textInstruction']);
        $inputTextHelpLink                      = esc_html(!empty(Config::getInstance()->get('textHelpLink')) ? Config::getInstance()->get('textHelpLink') : Config::getInstance()->defaultConfig()['textHelpLink']);
        $inputTextBackLink                      = esc_html(!empty(Config::getInstance()->get('textBackLink')) ? Config::getInstance()->get('textBackLink') : Config::getInstance()->defaultConfig()['textBackLink']);
        $inputTextCookieSettingSaved            = esc_html(!empty(Config::getInstance()->get('textCookieSettingSaved')) ? Config::getInstance()->get('textCookieSettingSaved') : Config::getInstance()->defaultConfig()['textCookieSettingSaved']);
        $textareaTextOptOut                     = esc_textarea(!empty(Config::getInstance()->get('textOptOut')) ? Config::getInstance()->get('textOptOut') : Config::getInstance()->defaultConfig()['textOptOut']);
        $textareaTextOptOutSuccess              = esc_textarea(!empty(Config::getInstance()->get('textOptOutSuccess')) ? Config::getInstance()->get('textOptOutSuccess') : Config::getInstance()->defaultConfig()['textOptOutSuccess']);
        $inputTextLogColumnTitleUserAction      = esc_html(!empty(Config::getInstance()->get('textLogColumnTitleUserAction')) ? Config::getInstance()->get('textLogColumnTitleUserAction') : Config::getInstance()->defaultConfig()['textLogColumnTitleUserAction']);
        $inputTextLogColumnTitleStamp           = esc_html(!empty(Config::getInstance()->get('textLogColumnTitleStamp')) ? Config::getInstance()->get('textLogColumnTitleStamp') : Config::getInstance()->defaultConfig()['textLogColumnTitleStamp']);
        $inputTextLogOptionThirdParty           = esc_html(!empty(Config::getInstance()->get('textLogOptionThirdParty')) ? Config::getInstance()->get('textLogOptionThirdParty') : Config::getInstance()->defaultConfig()['textLogOptionThirdParty']);
        $inputTextLogOptionFirstParty           = esc_html(!empty(Config::getInstance()->get('textLogOptionFirstParty')) ? Config::getInstance()->get('textLogOptionFirstParty') : Config::getInstance()->defaultConfig()['textLogOptionFirstParty']);
        $inputTextLogOptionRejectAllCookies     = esc_html(!empty(Config::getInstance()->get('textLogOptionRejectAllCookies')) ? Config::getInstance()->get('textLogOptionRejectAllCookies') : Config::getInstance()->defaultConfig()['textLogOptionRejectAllCookies']);
        $inputTextLogOptionNoSelection          = esc_html(!empty(Config::getInstance()->get('textLogOptionNoSelection')) ? Config::getInstance()->get('textLogOptionNoSelection') : Config::getInstance()->defaultConfig()['textLogOptionNoSelection']);
        $inputTextLogOptOutGoogleAnalytics      = esc_html(!empty(Config::getInstance()->get('textLogOptOutGoogleAnalytics')) ? Config::getInstance()->get('textLogOptOutGoogleAnalytics') : Config::getInstance()->defaultConfig()['textLogOptOutGoogleAnalytics']);
        $inputTextLogOptOutFacebookPixel        = esc_html(!empty(Config::getInstance()->get('textLogOptOutFacebookPixel')) ? Config::getInstance()->get('textLogOptOutFacebookPixel') : Config::getInstance()->defaultConfig()['textLogOptOutFacebookPixel']);
        $inputTextLogOptInUnblockType           = esc_html(!empty(Config::getInstance()->get('textLogOptInToUnblockType')) ? Config::getInstance()->get('textLogOptInToUnblockType') : Config::getInstance()->defaultConfig()['textLogOptInToUnblockType']);

        include Backend::getInstance()->templatePath.'/settings-texts.html';
    }

    /**
     * displayTabStyling function.
     *
     * @access public
     * @return void
     */
    public function displayTabStyling()
    {
        if (!empty($_POST['formSend']) && check_admin_referer('borlabs_cookie_styling')) {
            $this->saveStyling($_POST);

            Backend::getInstance()->addMessage(_x('Saved successfully.', 'Status message', 'borlabs-cookie'), 'success');
        }

        $inputStylingPrimaryBackgroundColor     = esc_html(Config::getInstance()->get('stylingPrimaryBackgroundColor'));
        $inputStylingSecondaryBackgroundColor   = esc_html(Config::getInstance()->get('stylingSecondaryBackgroundColor'));
        $inputStylingTextColor                  = esc_html(Config::getInstance()->get('stylingTextColor'));
        $inputStylingLinkColor                  = esc_html(Config::getInstance()->get('stylingLinkColor'));
        $inputStylingSuccessMessageColor        = esc_html(Config::getInstance()->get('stylingSuccessMessageColor'));
        $inputStylingCookieOptionBorderColor    = esc_html(Config::getInstance()->get('stylingCookieOptionBorderColor'));

        $inputStylingCookieOptionBackgroundColor        = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColor'));
        $inputStylingCookieOptionBackgroundColorHover   = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColorHover'));
        $inputStylingCookieOptionTextColor              = esc_html(Config::getInstance()->get('stylingCookieOptionTextColor'));
        $inputStylingCookieOptionTextColorHover         = esc_html(Config::getInstance()->get('stylingCookieOptionTextColorHover'));

        $inputStylingCookieOptionBackgroundColor2       = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColor2'));
        $inputStylingCookieOptionBackgroundColorHover2  = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColorHover2'));
        $inputStylingCookieOptionTextColor2             = esc_html(Config::getInstance()->get('stylingCookieOptionTextColor2'));
        $inputStylingCookieOptionTextColorHover2        = esc_html(Config::getInstance()->get('stylingCookieOptionTextColorHover2'));

        $inputStylingCookieOptionBackgroundColor3       = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColor3'));
        $inputStylingCookieOptionBackgroundColorHover3  = esc_html(Config::getInstance()->get('stylingCookieOptionBackgroundColorHover3'));
        $inputStylingCookieOptionTextColor3             = esc_html(Config::getInstance()->get('stylingCookieOptionTextColor3'));
        $inputStylingCookieOptionTextColorHover3        = esc_html(Config::getInstance()->get('stylingCookieOptionTextColorHover3'));

        $inputStylingBlockedContentBackgroundColor          = esc_html(Config::getInstance()->get('stylingBlockedContentBackgroundColor'));
        $inputStylingBlockedContentTextColor                = esc_html(Config::getInstance()->get('stylingBlockedContentTextColor'));
        $inputStylingBlockedContentButtonBackgroundColor    = esc_html(Config::getInstance()->get('stylingBlockedContentButtonBackgroundColor'));
        $inputStylingBlockedContentButtonBackgroundColorHover = esc_html(Config::getInstance()->get('stylingBlockedContentButtonBackgroundColorHover'));
        $inputStylingBlockedContentButtonTextColor          = esc_html(Config::getInstance()->get('stylingBlockedContentButtonTextColor'));
        $inputStylingBlockedContentButtonTextColorHover     = esc_html(Config::getInstance()->get('stylingBlockedContentButtonTextColorHover'));
        $inputStylingBlockedContentSuccessMessage           = esc_html(Config::getInstance()->get('stylingBlockedContentSuccessMessage'));

        $textareaStylingCustomCSS                           = esc_textarea(Config::getInstance()->get('stylingCustomCSS'));

        include Backend::getInstance()->templatePath.'/settings-styling.html';
    }

    /**
     * displayTabImportExport function.
     *
     * @access public
     * @return void
     */
    public function displayTabImportExport()
    {
        if (!empty($_POST['formSend']) && !empty($_POST['importSettings']) && check_admin_referer('borlabs_cookie_import_export')) {
            $this->importSettings($_POST['importSettings']);
        }

        $textareaExportSettings = esc_textarea(json_encode(Config::getInstance()->get()));

        include Backend::getInstance()->templatePath.'/settings-import-export.html';
    }

    /**
     * saveSettings function.
     *
     * @access public
     * @param mixed $data
     * @return void
     */
    public function saveSettings($data)
    {
        $updatedConfig = Config::getInstance()->get();

        $updatedConfig['cookieActivated']               = !empty($data['cookieActivated']) ? 'yes' : 'no';

        // Option 1
        $updatedConfig['codeCookieThirdParty']          = stripslashes($data['codeCookieThirdParty']);
        $updatedConfig['googleAnalyticsId']             = !empty($data['googleAnalyticsId']) ? stripslashes($data['googleAnalyticsId']) : '';
        $updatedConfig['facebookPixelId']               = !empty($data['facebookPixelId']) ? stripslashes($data['facebookPixelId']) : '';
        $updatedConfig['cookieLifetime']                = !empty($data['cookieLifetime']) && intval($data['cookieLifetime']) >= 1 ? intval($data['cookieLifetime']) : 365;
        $updatedConfig['cookiesForBots']                = !empty($data['cookiesForBots']) ? true : false;

        // Option 2
        $updatedConfig['showCookieOption2']             = !empty($data['showCookieOption2']) ? true : false;
        $updatedConfig['codeCookieFirstParty']          = stripslashes($data['codeCookieFirstParty']);
        $updatedConfig['cookieLifetimeOption2']         = !empty($data['cookieLifetimeOption2']) && intval($data['cookieLifetimeOption2']) >= 1 ? intval($data['cookieLifetimeOption2']) : 365;
        $updatedConfig['executeFirstPartyCodeOnThirdParty'] = !empty($data['executeFirstPartyCodeOnThirdParty']) ? true : false;

        // Option 3
        $updatedConfig['showCookieOption3']             = !empty($data['showCookieOption3']) ? true : false;
        $updatedConfig['cookieLifetimeOption3']         = !empty($data['cookieLifetimeOption3']) && intval($data['cookieLifetimeOption3']) >= 1  ? intval($data['cookieLifetimeOption3']) : 365;
        $updatedConfig['cookieRespectDoNotTrack']       = !empty($data['cookieRespectDoNotTrack']) ? true : false;

        // Cookie box settings
        $updatedConfig['cookieBoxShow']                 = 'automatic';

        if (!empty($data['cookieBoxShow'])) {
            if ($data['cookieBoxShow'] === 'always') {
                $updatedConfig['cookieBoxShow'] = 'always';
            }

            if ($data['cookieBoxShow'] === 'never') {
                $updatedConfig['cookieBoxShow'] = 'never';
            }
        }

        $updatedConfig['cookieBoxBlocks']               = !empty($data['cookieBoxBlocks']) ? true : false;

        $updatedConfig['cookieBoxIntegration']          = 'script';

        if (!empty($data['cookieBoxIntegration'])) {
            if ($data['cookieBoxIntegration'] === 'script') {
                $updatedConfig['cookieBoxIntegration'] = 'script';
            }

            if ($data['cookieBoxIntegration'] === 'html') {
                $updatedConfig['cookieBoxIntegration'] = 'html';
            }
        }

        $updatedConfig['cookieBoxPosition'] = 'top';

        if (!empty($data['cookieBoxPosition'])) {
            if ($data['cookieBoxPosition'] === 'center') {
                $updatedConfig['cookieBoxPosition'] = 'center';
            }

            if ($data['cookieBoxPosition'] === 'bottom') {
                $updatedConfig['cookieBoxPosition'] = 'bottom';
            }
        }

        $updatedConfig['privacyPageId']         = !empty($data['privacyPageId']) ? intval($data['privacyPageId']) : 0;
        $updatedConfig['imprintPageId']         = !empty($data['imprintPageId']) ? intval($data['imprintPageId']) : 0;

        $updatedConfig['hideCookieBoxOnPage']           = [];

        if (!empty($data['hideCookieBoxOnPage'])) {
            $data['hideCookieBoxOnPage'] = stripslashes($data['hideCookieBoxOnPage']);

            $data['hideCookieBoxOnPage'] = preg_split('/\r\n|[\r\n]/', $data['hideCookieBoxOnPage']);

            if (!empty($data['hideCookieBoxOnPage'])) {
                foreach ($data['hideCookieBoxOnPage'] as $path) {
                    $path = trim(stripslashes($path));

                    if (!empty($path)) {
                        $updatedConfig['hideCookieBoxOnPage'][] = $path;
                    }
                }
            }
        }

        // Multisite Network settings
        $siteURLInfo = parse_url(site_url());
        $networkDomain = $siteURLInfo['host'];

        $updatedConfig['cookieDomain']          = !empty($data['cookieDomain']) ? stripslashes($data['cookieDomain']) : $networkDomain;
        $updatedConfig['cookiePath']            = !empty($data['cookiePath']) ? stripslashes($data['cookiePath']) : '/';
        $updatedConfig['aggregateCookieConsent']= !empty($data['aggregateCookieConsent']) ? true : false;

        // Iframe block settings
        $updatedConfig['removeIframesInFeeds']  = !empty($data['removeIframesInFeeds']) ? true : false;

        if (!empty($updatedConfig['privacyPageId'])) {
            $postData = get_post($updatedConfig['privacyPageId']);

            if (!empty($postData->ID)) {
                $permalink = get_permalink($postData->ID);

                $updatedConfig['privacyPageTitle'] = $postData->post_title;
                $updatedConfig['privacyPageURL'] = !empty($permalink) ? $permalink : $postData->guid;
            }
        }

        if (!empty($updatedConfig['imprintPageId'])) {
            $postData = get_post($updatedConfig['imprintPageId']);

            if (!empty($postData->ID)) {
                $permalink = get_permalink($postData->ID);

                $updatedConfig['imprintPageTitle'] = $postData->post_title;
                $updatedConfig['imprintPageURL'] = !empty($permalink) ? $permalink : $postData->guid;
            }
        }

        // Save config
        Config::getInstance()->saveConfig($updatedConfig);
    }

    /**
     * saveTexts function.
     *
     * @access public
     * @param mixed $data
     * @return void
     */
    public function saveTexts($data)
    {
        $defaultConfig = Config::getInstance()->defaultConfig();

        $updatedConfig = Config::getInstance()->get();

        $updatedConfig['textCookieOptionThirdParty']        = !empty($data['textCookieOptionThirdParty'])       ? stripslashes($data['textCookieOptionThirdParty']) : $defaultConfig['textCookieOptionThirdParty'];
        $updatedConfig['textCookieOptionFirstParty']        = !empty($data['textCookieOptionFirstParty'])       ? stripslashes($data['textCookieOptionFirstParty']) : $defaultConfig['textCookieOptionFirstParty'];
        $updatedConfig['textCookieOptionRejectThirdParty']  = !empty($data['textCookieOptionRejectThirdParty']) ? stripslashes($data['textCookieOptionRejectThirdParty']) : $defaultConfig['textCookieOptionRejectThirdParty'];
        $updatedConfig['textHelpHeadline']                  = !empty($data['textHelpHeadline'])                 ? stripslashes($data['textHelpHeadline']) : $defaultConfig['textHelpHeadline'];
        $updatedConfig['textHelpTop']                       = !empty($data['textHelpTop'])                      ? stripslashes($data['textHelpTop']) : $defaultConfig['textHelpTop'];
        $updatedConfig['textHelpCookieOptionThirdParty']    = !empty($data['textHelpCookieOptionThirdParty'])   ? stripslashes($data['textHelpCookieOptionThirdParty']) : $defaultConfig['textHelpCookieOptionThirdParty'];
        $updatedConfig['textHelpCookieOptionFirstParty']    = !empty($data['textHelpCookieOptionFirstParty'])   ? stripslashes($data['textHelpCookieOptionFirstParty']) : $defaultConfig['textHelpCookieOptionFirstParty'];
        $updatedConfig['textHelpCookieOptionRejectThirdParty'] = !empty($data['textHelpCookieOptionRejectThirdParty']) ? stripslashes($data['textHelpCookieOptionRejectThirdParty']) : $defaultConfig['textHelpCookieOptionRejectThirdParty'];
        $updatedConfig['textHelpBottom']                    = !empty($data['textHelpBottom'])                   ? stripslashes($data['textHelpBottom']) : $defaultConfig['textHelpBottom'];
        $updatedConfig['textTitle']                         = !empty($data['textTitle'])                        ? stripslashes($data['textTitle']) : $defaultConfig['textTitle'];
        $updatedConfig['textExplanation']                   = !empty($data['textExplanation'])                  ? stripslashes($data['textExplanation']) : $defaultConfig['textExplanation'];
        $updatedConfig['textInstruction']                   = !empty($data['textInstruction'])                  ? stripslashes($data['textInstruction']) : $defaultConfig['textInstruction'];
        $updatedConfig['textHelpLink']                      = !empty($data['textHelpLink'])                     ? stripslashes($data['textHelpLink']) : $defaultConfig['textHelpLink'];
        $updatedConfig['textBackLink']                      = !empty($data['textBackLink'])                     ? stripslashes($data['textBackLink']) : $defaultConfig['textBackLink'];
        $updatedConfig['textCookieSettingSaved']            = !empty($data['textCookieSettingSaved'])           ? stripslashes($data['textCookieSettingSaved']) : $defaultConfig['textCookieSettingSaved'];
        $updatedConfig['textOptOut']                        = !empty($data['textOptOut'])                       ? stripslashes($data['textOptOut']) : $defaultConfig['textOptOut'];
        $updatedConfig['textOptOutSuccess']                 = !empty($data['textOptOutSuccess'])                ? stripslashes($data['textOptOutSuccess']) : $defaultConfig['textOptOutSuccess'];
        $updatedConfig['textLogColumnTitleUserAction']      = !empty($data['textLogColumnTitleUserAction'])      ? stripslashes($data['textLogColumnTitleUserAction']) : $defaultConfig['textLogColumnTitleUserAction'];
        $updatedConfig['textLogColumnTitleStamp']           = !empty($data['textLogColumnTitleStamp'])           ? stripslashes($data['textLogColumnTitleStamp']) : $defaultConfig['textLogColumnTitleStamp'];
        $updatedConfig['textLogOptionThirdParty']           = !empty($data['textLogOptionThirdParty'])          ? stripslashes($data['textLogOptionThirdParty']) : $defaultConfig['textLogOptionThirdParty'];
        $updatedConfig['textLogOptionFirstParty']           = !empty($data['textLogOptionFirstParty'])          ? stripslashes($data['textLogOptionFirstParty']) : $defaultConfig['textLogOptionFirstParty'];
        $updatedConfig['textLogOptionRejectAllCookies']     = !empty($data['textLogOptionRejectAllCookies'])    ? stripslashes($data['textLogOptionRejectAllCookies']) : $defaultConfig['textLogOptionRejectAllCookies'];
        $updatedConfig['textLogOptionNoSelection']          = !empty($data['textLogOptionNoSelection'])         ? stripslashes($data['textLogOptionNoSelection']) : $defaultConfig['textLogOptionNoSelection'];
        $updatedConfig['textLogOptOutGoogleAnalytics']      = !empty($data['textLogOptOutGoogleAnalytics'])     ? stripslashes($data['textLogOptOutGoogleAnalytics']) : $defaultConfig['textLogOptOutGoogleAnalytics'];
        $updatedConfig['textLogOptOutFacebookPixel']        = !empty($data['textLogOptOutFacebookPixel'])       ? stripslashes($data['textLogOptOutFacebookPixel']) : $defaultConfig['textLogOptOutFacebookPixel'];
        $updatedConfig['textLogOptInToUnblockType']         = !empty($data['textLogOptInToUnblockType'])       ? stripslashes($data['textLogOptInToUnblockType']) : $defaultConfig['textLogOptInToUnblockType'];

        // Save config
        Config::getInstance()->saveConfig($updatedConfig);
    }

    public function saveStyling($data)
    {
        $defaultConfig = Config::getInstance()->defaultConfig();

        $updatedConfig = Config::getInstance()->get();

        $updatedConfig['stylingPrimaryBackgroundColor']                     = !empty($data['stylingPrimaryBackgroundColor']) && $this->validateHexColor($data['stylingPrimaryBackgroundColor']) ? stripslashes($data['stylingPrimaryBackgroundColor']) : $defaultConfig['stylingPrimaryBackgroundColor'];
        $updatedConfig['stylingSecondaryBackgroundColor']                   = !empty($data['stylingSecondaryBackgroundColor']) && $this->validateHexColor($data['stylingSecondaryBackgroundColor']) ? stripslashes($data['stylingSecondaryBackgroundColor']) : $defaultConfig['stylingSecondaryBackgroundColor'];
        $updatedConfig['stylingTextColor']                                  = !empty($data['stylingTextColor']) && $this->validateHexColor($data['stylingTextColor']) ? stripslashes($data['stylingTextColor']) : $defaultConfig['stylingTextColor'];
        $updatedConfig['stylingLinkColor']                                  = !empty($data['stylingLinkColor']) && $this->validateHexColor($data['stylingLinkColor']) ? stripslashes($data['stylingLinkColor']) : $defaultConfig['stylingLinkColor'];
        $updatedConfig['stylingSuccessMessageColor']                        = !empty($data['stylingSuccessMessageColor']) && $this->validateHexColor($data['stylingSuccessMessageColor']) ? stripslashes($data['stylingSuccessMessageColor']) : $defaultConfig['stylingSuccessMessageColor'];
        $updatedConfig['stylingSuccessMessageColor']                        = !empty($data['stylingSuccessMessageColor']) && $this->validateHexColor($data['stylingSuccessMessageColor']) ? stripslashes($data['stylingSuccessMessageColor']) : $defaultConfig['stylingSuccessMessageColor'];
        $updatedConfig['stylingCookieOptionBorderColor']                    = !empty($data['stylingCookieOptionBorderColor']) && $this->validateHexColor($data['stylingCookieOptionBorderColor']) ? stripslashes($data['stylingCookieOptionBorderColor']) : $defaultConfig['stylingCookieOptionBorderColor'];

        $updatedConfig['stylingCookieOptionBackgroundColor']                = !empty($data['stylingCookieOptionBackgroundColor']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColor']) ? stripslashes($data['stylingCookieOptionBackgroundColor']) : $defaultConfig['stylingCookieOptionBackgroundColor'];
        $updatedConfig['stylingCookieOptionBackgroundColorHover']           = !empty($data['stylingCookieOptionBackgroundColorHover']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColorHover']) ? stripslashes($data['stylingCookieOptionBackgroundColorHover']) : $defaultConfig['stylingCookieOptionBackgroundColorHover'];
        $updatedConfig['stylingCookieOptionTextColor']                      = !empty($data['stylingCookieOptionTextColor']) && $this->validateHexColor($data['stylingCookieOptionTextColor']) ? stripslashes($data['stylingCookieOptionTextColor']) : $defaultConfig['stylingCookieOptionTextColor'];
        $updatedConfig['stylingCookieOptionTextColorHover']                 = !empty($data['stylingCookieOptionTextColorHover']) && $this->validateHexColor($data['stylingCookieOptionTextColorHover']) ? stripslashes($data['stylingCookieOptionTextColorHover']) : $defaultConfig['stylingCookieOptionTextColorHover'];

        $updatedConfig['stylingCookieOptionBackgroundColor2']               = !empty($data['stylingCookieOptionBackgroundColor2']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColor2']) ? stripslashes($data['stylingCookieOptionBackgroundColor2']) : $defaultConfig['stylingCookieOptionBackgroundColor2'];
        $updatedConfig['stylingCookieOptionBackgroundColorHover2']          = !empty($data['stylingCookieOptionBackgroundColorHover2']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColorHover2']) ? stripslashes($data['stylingCookieOptionBackgroundColorHover2']) : $defaultConfig['stylingCookieOptionBackgroundColorHover2'];
        $updatedConfig['stylingCookieOptionTextColor2']                     = !empty($data['stylingCookieOptionTextColor2']) && $this->validateHexColor($data['stylingCookieOptionTextColor2']) ? stripslashes($data['stylingCookieOptionTextColor2']) : $defaultConfig['stylingCookieOptionTextColor2'];
        $updatedConfig['stylingCookieOptionTextColorHover2']                = !empty($data['stylingCookieOptionTextColorHover2']) && $this->validateHexColor($data['stylingCookieOptionTextColorHover2']) ? stripslashes($data['stylingCookieOptionTextColorHover2']) : $defaultConfig['stylingCookieOptionTextColorHover2'];

        $updatedConfig['stylingCookieOptionBackgroundColor3']               = !empty($data['stylingCookieOptionBackgroundColor3']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColor3']) ? stripslashes($data['stylingCookieOptionBackgroundColor3']) : $defaultConfig['stylingCookieOptionBackgroundColor3'];
        $updatedConfig['stylingCookieOptionBackgroundColorHover3']          = !empty($data['stylingCookieOptionBackgroundColorHover3']) && $this->validateHexColor($data['stylingCookieOptionBackgroundColorHover3']) ? stripslashes($data['stylingCookieOptionBackgroundColorHover3']) : $defaultConfig['stylingCookieOptionBackgroundColorHover3'];
        $updatedConfig['stylingCookieOptionTextColor3']                     = !empty($data['stylingCookieOptionTextColor3']) && $this->validateHexColor($data['stylingCookieOptionTextColor3']) ? stripslashes($data['stylingCookieOptionTextColor3']) : $defaultConfig['stylingCookieOptionTextColor3'];
        $updatedConfig['stylingCookieOptionTextColorHover3']                = !empty($data['stylingCookieOptionTextColorHover3']) && $this->validateHexColor($data['stylingCookieOptionTextColorHover3']) ? stripslashes($data['stylingCookieOptionTextColorHover3']) : $defaultConfig['stylingCookieOptionTextColorHover3'];

        $updatedConfig['stylingBlockedContentBackgroundColor']              = !empty($data['stylingBlockedContentBackgroundColor']) && $this->validateHexColor($data['stylingBlockedContentBackgroundColor']) ? stripslashes($data['stylingBlockedContentBackgroundColor']) : $defaultConfig['stylingBlockedContentBackgroundColor'];
        $updatedConfig['stylingBlockedContentTextColor']                    = !empty($data['stylingBlockedContentTextColor']) && $this->validateHexColor($data['stylingBlockedContentTextColor']) ? stripslashes($data['stylingBlockedContentTextColor']) : $defaultConfig['stylingBlockedContentTextColor'];
        $updatedConfig['stylingBlockedContentButtonBackgroundColor']        = !empty($data['stylingBlockedContentButtonBackgroundColor']) && $this->validateHexColor($data['stylingBlockedContentButtonBackgroundColor']) ? stripslashes($data['stylingBlockedContentButtonBackgroundColor']) : $defaultConfig['stylingBlockedContentButtonBackgroundColor'];
        $updatedConfig['stylingBlockedContentButtonBackgroundColorHover']   = !empty($data['stylingBlockedContentButtonBackgroundColorHover']) && $this->validateHexColor($data['stylingBlockedContentButtonBackgroundColorHover']) ? stripslashes($data['stylingBlockedContentButtonBackgroundColorHover']) : $defaultConfig['stylingBlockedContentButtonBackgroundColorHover'];
        $updatedConfig['stylingBlockedContentButtonTextColor']              = !empty($data['stylingBlockedContentButtonTextColor']) && $this->validateHexColor($data['stylingBlockedContentButtonTextColor']) ? stripslashes($data['stylingBlockedContentButtonTextColor']) : $defaultConfig['stylingBlockedContentButtonTextColor'];
        $updatedConfig['stylingBlockedContentButtonTextColorHover']         = !empty($data['stylingBlockedContentButtonTextColorHover']) && $this->validateHexColor($data['stylingBlockedContentButtonTextColorHover']) ? stripslashes($data['stylingBlockedContentButtonTextColorHover']) : $defaultConfig['stylingBlockedContentButtonTextColorHover'];
        $updatedConfig['stylingBlockedContentSuccessMessage']               = !empty($data['stylingBlockedContentSuccessMessage']) && $this->validateHexColor($data['stylingBlockedContentSuccessMessage']) ? stripslashes($data['stylingBlockedContentSuccessMessage']) : $defaultConfig['stylingBlockedContentSuccessMessage'];

        $updatedConfig['stylingCustomCSS']                                  = !empty($data['stylingCustomCSS']) ? stripslashes($data['stylingCustomCSS']) : $defaultConfig['stylingCustomCSS'];

        $pluginPath = realpath(dirname(__FILE__).'/../../../');

        if (file_exists(WP_CONTENT_DIR.'/cache/borlabs_cookie')) {
            $css = $this->getFinalCustomCSS($updatedConfig);

            $mainCSS = file_get_contents($pluginPath.'/css/borlabs-cookie.css');
            file_put_contents(
                WP_CONTENT_DIR.'/cache/borlabs_cookie/borlabs-cookie_'.get_current_blog_id().'_'.Multilanguage::getInstance()->getCurrentLanguageCode().'.css',
                $css.$mainCSS.preg_replace("/(\r|\n|\t){1,}/", " ", $updatedConfig['stylingCustomCSS'])
            );
        }

        // Save config
        Config::getInstance()->saveConfig($updatedConfig);

        // Update style version
        $styleVersion = get_option('BorlabsCookieStyleVersion_'.Multilanguage::getInstance()->getCurrentLanguageCode(), 1);
        $styleVersion = intval($styleVersion)+1;
        update_option('BorlabsCookieStyleVersion_'.Multilanguage::getInstance()->getCurrentLanguageCode(), $styleVersion, false);
    }

    /**
     * importSettings function.
     *
     * @access public
     * @param mixed $settings
     * @return void
     */
    public function importSettings($settings)
    {
        $settings = stripslashes($settings);

        // Check if settings is json
        if (!Tools::getinstance()->isStringJSON($settings)) {
            Backend::getInstance()->addMessage(_x('Your settings are not valid JSON.', 'Status message', 'borlabs-cookie'), 'success');
        } else {
            $data = json_decode($settings, true);

            $this->saveSettings($data);
            $this->saveTexts($data);
            $this->saveStyling($data);

            Backend::getInstance()->addMessage(_x('Import successfully.', 'Status message', 'borlabs-cookie'), 'success');
        }
    }

    /**
     * validateHexColor function.
     *
     * @access public
     * @param mixed $color
     * @return void
     */
    public function validateHexColor($color)
    {
        return preg_match('/#([a-f0-9]{3}){1,2}\b/i', $color);
    }

    /**
     * getCustomCSS function.
     *
     * @access public
     * @return void
     */
    public function getCustomCSSTemplate()
    {
        $css = <<<EOT
[data-borlabs-cookie-wrap] [data-borlabs-cookie] {background:{stylingPrimaryBackgroundColor} !important; color:{stylingTextColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] p {color:{stylingTextColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] p a {color:{stylingLinkColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] p.borlabsCookieSubHeadline {background:{stylingSecondaryBackgroundColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] p.borlabsCookieSuccess {color:{stylingSuccessMessageColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul {border-top: 1px solid {stylingCookieOptionBorderColor} !important; border-left: 1px solid {stylingCookieOptionBorderColor} !important; border-right: 1px solid {stylingCookieOptionBorderColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption1 {background:{stylingCookieOptionBackgroundColor} !important; border-bottom: 1px solid {stylingCookieOptionBorderColor} !important; color:{stylingCookieOptionTextColor} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption1:hover {background-color:{stylingCookieOptionBackgroundColorHover} !important; color:{stylingCookieOptionTextColorHover} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption2 {background:{stylingCookieOptionBackgroundColor2} !important; border-bottom: 1px solid {stylingCookieOptionBorderColor} !important; color:{stylingCookieOptionTextColor2} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption2:hover {background-color:{stylingCookieOptionBackgroundColorHover2} !important; color:{stylingCookieOptionTextColorHover2} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption3 {background:{stylingCookieOptionBackgroundColor3} !important; border-bottom: 1px solid {stylingCookieOptionBorderColor} !important; color:{stylingCookieOptionTextColor3} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-selection] ul li.cookieOption3:hover {background-color:{stylingCookieOptionBackgroundColorHover3} !important; color:{stylingCookieOptionTextColorHover3} !important;}
[data-borlabs-cookie-wrap] [data-borlabs-cookie] [data-borlabs-help] a {color:{stylingLinkColor} !important;}
.borlabs-cookie-bct-default, [data-borlabs-cookie-opt-out] {background:{stylingBlockedContentBackgroundColor} !important; color: {stylingBlockedContentTextColor} !important;}
.borlabs-cookie-bct-default p, [data-borlabs-cookie-opt-out] p {color: {stylingBlockedContentTextColor} !important;}
.borlabs-cookie-bct-default a[role="button"], [data-borlabs-cookie-opt-out] a[role="button"] {background:{stylingBlockedContentButtonBackgroundColor} !important; color: {stylingBlockedContentButtonTextColor} !important;}
.borlabs-cookie-bct-default a[role="button"]:hover, [data-borlabs-cookie-opt-out] a[role="button"]:hover {background:{stylingBlockedContentButtonBackgroundColorHover} !important; color: {stylingBlockedContentButtonTextColorHover} !important;}
[data-borlabs-cookie-opt-out] [data-borlabs-cookie-opt-out-success] {color: {stylingBlockedContentSuccessMessage} !important;}
EOT;
        return $css;
    }

    /**
     * getFinalCustomCSS function.
     *
     * @access public
     * @param mixed $styles
     * @return void
     */
    public function getFinalCustomCSS($styles)
    {
        $css = str_replace(
            [
                '{stylingPrimaryBackgroundColor}',
                '{stylingSecondaryBackgroundColor}',
                '{stylingTextColor}',
                '{stylingLinkColor}',
                '{stylingSuccessMessageColor}',
                '{stylingCookieOptionBorderColor}',

                '{stylingCookieOptionBackgroundColor}',
                '{stylingCookieOptionBackgroundColorHover}',
                '{stylingCookieOptionTextColor}',
                '{stylingCookieOptionTextColorHover}',

                '{stylingCookieOptionBackgroundColor2}',
                '{stylingCookieOptionBackgroundColorHover2}',
                '{stylingCookieOptionTextColor2}',
                '{stylingCookieOptionTextColorHover2}',

                '{stylingCookieOptionBackgroundColor3}',
                '{stylingCookieOptionBackgroundColorHover3}',
                '{stylingCookieOptionTextColor3}',
                '{stylingCookieOptionTextColorHover3}',

                '{stylingBlockedContentBackgroundColor}',
                '{stylingBlockedContentTextColor}',
                '{stylingBlockedContentButtonBackgroundColor}',
                '{stylingBlockedContentButtonBackgroundColorHover}',
                '{stylingBlockedContentButtonTextColor}',
                '{stylingBlockedContentButtonTextColorHover}',
                '{stylingBlockedContentSuccessMessage}',
            ],
            [
                $styles['stylingPrimaryBackgroundColor'],
                $styles['stylingSecondaryBackgroundColor'],
                $styles['stylingTextColor'],
                $styles['stylingLinkColor'],
                $styles['stylingSuccessMessageColor'],
                $styles['stylingCookieOptionBorderColor'],

                $styles['stylingCookieOptionBackgroundColor'],
                $styles['stylingCookieOptionBackgroundColorHover'],
                $styles['stylingCookieOptionTextColor'],
                $styles['stylingCookieOptionTextColorHover'],

                $styles['stylingCookieOptionBackgroundColor2'],
                $styles['stylingCookieOptionBackgroundColorHover2'],
                $styles['stylingCookieOptionTextColor2'],
                $styles['stylingCookieOptionTextColorHover2'],

                $styles['stylingCookieOptionBackgroundColor3'],
                $styles['stylingCookieOptionBackgroundColorHover3'],
                $styles['stylingCookieOptionTextColor3'],
                $styles['stylingCookieOptionTextColorHover3'],

                $styles['stylingBlockedContentBackgroundColor'],
                $styles['stylingBlockedContentTextColor'],
                $styles['stylingBlockedContentButtonBackgroundColor'],
                $styles['stylingBlockedContentButtonBackgroundColorHover'],
                $styles['stylingBlockedContentButtonTextColor'],
                $styles['stylingBlockedContentButtonTextColorHover'],
                $styles['stylingBlockedContentSuccessMessage'],
            ],
            $this->getCustomCSSTemplate()
        );

        return $css;
    }
}
